<?php
namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\web\Response;
use app\models\User;
use app\models\MasterInfo;
use app\models\Service;
use app\models\Appointment;

class AppointmentController extends Controller
{
    /**
     * Страница создания заявки
     */
    public function actionCreate()
    {
        if (Yii::$app->user->isGuest) {
            Yii::$app->session->setFlash('error', 'Нужно авторизоваться!');
            return $this->redirect(['site/login']);
        }
        $masters = User::find()
            ->where(['is_admin' => 1])
            ->all();

        return $this->render('create', [
            'masters' => $masters,
        ]);
    }

    /**
     * AJAX: Получить услуги мастера
     */
    public function actionGetServices($master_id)
    {
        Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;

        // Находим master_info для этого мастера
        $masterInfo = MasterInfo::find()
            ->where(['user_id' => $master_id])
            ->one();

        if (!$masterInfo) {
            return ['error' => 'MasterInfo не найден'];
        }

        // Получаем услуги через master_services (там где проставлены галочки)
        $services = Service::find()
            ->alias('s')
            ->select(['s.id', 's.name', 's.description', 's.duration', 's.price', 'ms.price_modifier'])
            ->innerJoin('master_services ms', 'ms.service_id = s.id')
            ->where(['ms.master_info_id' => $masterInfo->id])
            ->andWhere(['s.is_active' => 1])
            ->asArray()
            ->all();

        // Добавляем итоговую цену
        foreach ($services as &$service) {
            $service['final_price'] = $service['price'] + floatval($service['price_modifier'] ?? 0);
        }

        return $services;
    }

    /**
     * AJAX: Получить свободное время мастера
     */
    public function actionGetFreeTime($master_id, $date, $service_id)
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        try {
            // Получаем услугу
            $service = Service::findOne($service_id);
            if (!$service) {
                return [];
            }

            // Находим master_info_id для мастера
            $masterInfo = MasterInfo::find()
                ->where(['user_id' => $master_id])
                ->one();

            if (!$masterInfo) {
                // Если нет master_info, возвращаем все слоты (тестовый режим)
                return $this->generateAllSlots($date, $service->duration);
            }

            // Получаем занятые записи
            $startOfDay = $date . ' 00:00:00';
            $endOfDay = $date . ' 23:59:59';

            $busyTimes = Appointment::find()
                ->where(['master_info_id' => $masterInfo->id])
                ->andWhere(['between', 'start_time', $startOfDay, $endOfDay])
                ->all();

            // Генерируем слоты с 9 до 20
            $freeSlots = [];
            $workStart = 9;
            $workEnd = 20;

            $currentTime = strtotime($date . ' ' . $workStart . ':00:00');
            $endTime = strtotime($date . ' ' . $workEnd . ':00:00');

            while ($currentTime + $service->duration * 60 <= $endTime) {
                $slotStart = $currentTime;
                $slotEnd = $currentTime + $service->duration * 60;

                // Проверяем занятость
                $isFree = true;
                foreach ($busyTimes as $busy) {
                    $busyStart = strtotime($busy->start_time);
                    $busyEnd = strtotime($busy->end_time);

                    if ($slotStart < $busyEnd && $slotEnd > $busyStart) {
                        $isFree = false;
                        break;
                    }
                }

                if ($isFree) {
                    $freeSlots[] = [
                        'time' => date('H:i', $slotStart),
                        'start' => date('Y-m-d H:i:s', $slotStart)
                    ];
                }

                $currentTime += 30 * 60;
            }

            return $freeSlots;

        } catch (\Exception $e) {
            // В случае ошибки возвращаем тестовые слоты
            return $this->generateAllSlots($date, 60);
        }
    }

// Вспомогательный метод для теста
    private function generateAllSlots($date, $duration)
    {
        $slots = [];
        $workStart = 9;
        $workEnd = 20;

        $currentTime = strtotime($date . ' ' . $workStart . ':00:00');
        $endTime = strtotime($date . ' ' . $workEnd . ':00:00');

        while ($currentTime + $duration * 60 <= $endTime) {
            $slots[] = [
                'time' => date('H:i', $currentTime),
                'start' => date('Y-m-d H:i:s', $currentTime)
            ];
            $currentTime += 30 * 60;
        }

        return $slots;
    }

    /**
     * Сохранение заявки
     */
    public function actionSave()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        try {
            if (Yii::$app->user->isGuest) {
                return ['success' => false, 'message' => 'Необходимо авторизоваться'];
            }

            $post = Yii::$app->request->post();

            // Проверяем обязательные поля
            if (empty($post['master_id']) || empty($post['service_id']) || empty($post['start_time']) || empty($post['payment_method'])) {
                return ['success' => false, 'message' => 'Заполните все обязательные поля'];
            }

            // Получаем услугу
            $service = Service::findOne($post['service_id']);
            if (!$service) {
                return ['success' => false, 'message' => 'Услуга не найдена'];
            }

            // Находим master_info_id
            $masterInfo = MasterInfo::find()
                ->where(['user_id' => $post['master_id']])
                ->one();

            if (!$masterInfo) {
                // Создаем если нет
                $masterInfo = new MasterInfo();
                $masterInfo->user_id = $post['master_id'];
                $masterInfo->is_active = 1;
                $masterInfo->save();
            }

            // Создаем запись
            $appointment = new Appointment();
            $appointment->master_info_id = $masterInfo->id;
            $appointment->service_id = $post['service_id'];
            $appointment->client_id = Yii::$app->user->id;
            $appointment->start_time = $post['start_time'];
            $appointment->end_time = date('Y-m-d H:i:s', strtotime($post['start_time']) + $service->duration * 60);
            $appointment->payment_method = $post['payment_method'];
            $appointment->additional_info = !empty($post['additional_info']) ? $post['additional_info'] : '';
            $appointment->status = 0; // pending

            if ($appointment->save()) {
                return ['success' => true, 'message' => 'Запись успешно создана!'];
            } else {
                return ['success' => false, 'errors' => $appointment->errors];
            }

        } catch (\Exception $e) {
            return ['success' => false, 'message' => 'Ошибка: ' . $e->getMessage()];
        }
    }
}