<?php
namespace app\controllers;

use app\models\Appointment;
use app\models\MasterPortfolio;
use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\web\NotFoundHttpException;
use yii\web\UploadedFile;
use app\models\MasterInfo;
use app\models\MasterSchedule;
use app\models\MasterService;
use app\models\Service;
use app\models\User;

class MasterController extends Controller
{
    public function behaviors()
    {
        return [
            'access' => [
                'class' => \yii\filters\AccessControl::class,
                'only' => ['index', 'profile', 'services', 'schedule', 'portfolio', 'appointments', 'view'],
                'rules' => [
                    [
                        'actions' => ['view'], // Страница мастера доступна всем
                        'allow' => true,
                        'roles' => ['?', '@'], // ? - гости, @ - авторизованные
                    ],
                    [
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function ($rule, $action) {
                            // Только мастера (is_admin = 1)
                            return Yii::$app->user->identity->is_admin == 1;
                        }
                    ],
                ],
            ],
        ];
    }

    /**
     * Dashboard мастера
     */
    public function actionIndex()
    {
        $user = Yii::$app->user->identity;
        $masterInfo = MasterInfo::find()->where(['user_id' => $user->id])->one();

        // Если нет информации о мастере - создаем
        if (!$masterInfo) {
            $masterInfo = new MasterInfo();
            $masterInfo->user_id = $user->id;
            $masterInfo->is_active = 1;
            $masterInfo->save();
        }

        return $this->render('index', [
            'masterInfo' => $masterInfo,
            'user' => $user,
        ]);
    }

    /**
     * Редактирование профиля
     */
    public function actionProfile()
    {
        $user = Yii::$app->user->identity;
        $masterInfo = MasterInfo::find()->where(['user_id' => $user->id])->one();

        if (!$masterInfo) {
            $masterInfo = new MasterInfo();
            $masterInfo->user_id = $user->id;
        }

        if ($masterInfo->load(Yii::$app->request->post())) {

            // Загрузка фото сертификата
            $certificateFile = UploadedFile::getInstance($masterInfo, 'certificate_photo');
            if ($certificateFile) {
                $fileName = 'cert_' . $user->id . '_' . time() . '.' . $certificateFile->extension;
                $filePath = 'uploads/certificates/' . $fileName;

                if (!is_dir('uploads/certificates')) {
                    mkdir('uploads/certificates', 0777, true);
                }

                if ($certificateFile->saveAs($filePath)) {
                    $masterInfo->certificate_photo = $filePath;
                }
            }

            if ($masterInfo->save()) {
                Yii::$app->session->setFlash('success', 'Профиль обновлен');
                return $this->redirect(['index']);
            }
        }

        return $this->render('profile', [
            'masterInfo' => $masterInfo,
            'user' => $user,
        ]);
    }

    /**
     * Управление услугами мастера
     */
    public function actionServices()
    {
        $user = Yii::$app->user->identity;

        // Получаем master_info
        $masterInfo = MasterInfo::find()->where(['user_id' => $user->id])->one();
        if (!$masterInfo) {
            $masterInfo = new MasterInfo();
            $masterInfo->user_id = $user->id;
            $masterInfo->is_active = 1;
            $masterInfo->save();
        }

        // Обработка отправки формы
        if (Yii::$app->request->isPost) {
            $services = Yii::$app->request->post('service', []);

            // Удаляем старые услуги
            MasterService::deleteAll(['master_info_id' => $masterInfo->id]);

            // Добавляем новые
            foreach ($services as $serviceId => $data) {
                if (isset($data['selected']) && $data['selected']) {
                    $ms = new MasterService();
                    $ms->master_info_id = $masterInfo->id;
                    $ms->service_id = $serviceId;
                    $ms->price_modifier = !empty($data['modifier']) ? $data['modifier'] : 0;
                    $ms->save();
                }
            }

            Yii::$app->session->setFlash('success', 'Услуги сохранены');
            return $this->redirect(['services']);
        }

        // Получаем все услуги
        $allServices = Service::find()->all();

        // Получаем выбранные услуги мастера
        $selectedServices = MasterService::find()
            ->where(['master_info_id' => $masterInfo->id])
            ->indexBy('service_id')
            ->all();

        return $this->render('services', [
            'allServices' => $allServices,
            'selectedServices' => $selectedServices,
        ]);
    }

    /**
     * Расписание работы
     */
    public function actionSchedule()
    {
        $user = Yii::$app->user->identity;

        // Получаем master_info
        $masterInfo = MasterInfo::find()->where(['user_id' => $user->id])->one();
        if (!$masterInfo) {
            $masterInfo = new MasterInfo();
            $masterInfo->user_id = $user->id;
            $masterInfo->is_active = 1;
            $masterInfo->save();
        }

        $days = [
            1 => 'Понедельник',
            2 => 'Вторник',
            3 => 'Среда',
            4 => 'Четверг',
            5 => 'Пятница',
            6 => 'Суббота',
            7 => 'Воскресенье',
        ];

        // Обработка POST запроса
        if (Yii::$app->request->isPost) {
            $scheduleData = Yii::$app->request->post('schedule', []);

            // Удаляем старое расписание
            MasterSchedule::deleteAll(['master_info_id' => $masterInfo->id]);

            // Добавляем новое
            $savedCount = 0;
            foreach ($scheduleData as $day => $data) {
                if (isset($data['is_working']) && $data['is_working']) {
                    $schedule = new MasterSchedule();
                    $schedule->master_info_id = $masterInfo->id;
                    $schedule->day_of_week = $day;
                    $schedule->start_time = !empty($data['start_time']) ? $data['start_time'] : '09:00';
                    $schedule->end_time = !empty($data['end_time']) ? $data['end_time'] : '20:00';
                    $schedule->is_working = 1;

                    if ($schedule->save()) {
                        $savedCount++;
                    } else {
                        Yii::$app->session->setFlash('error', 'Ошибка при сохранении дня ' . $days[$day]);
                        return $this->redirect(['schedule']);
                    }
                }
            }

            Yii::$app->session->setFlash('success', "Расписание сохранено! Отмечено дней: $savedCount");
            return $this->redirect(['schedule']);
        }

        // Получаем текущее расписание
        $schedules = MasterSchedule::find()
            ->where(['master_info_id' => $masterInfo->id])
            ->indexBy('day_of_week')
            ->all();

        return $this->render('schedule', [
            'masterInfo' => $masterInfo,
            'days' => $days,
            'schedules' => $schedules,
        ]);
    }

    /**
     * Портфолио мастера (фото работ)
     */
    public function actionPortfolio()
    {
        $user = Yii::$app->user->identity;
        $masterInfo = MasterInfo::find()->where(['user_id' => $user->id])->one();

        if (!$masterInfo) {
            $masterInfo = new MasterInfo();
            $masterInfo->user_id = $user->id;
            $masterInfo->is_active = 1;
            $masterInfo->save();
        }

        // Получаем все фото портфолио
        $portfolio = MasterPortfolio::find()
            ->where(['master_info_id' => $masterInfo->id])
            ->orderBy(['created_at' => SORT_DESC])
            ->all();

        return $this->render('portfolio', [
            'masterInfo' => $masterInfo,
            'portfolio' => $portfolio,
        ]);
    }

    /**
     * Мои записи (клиенты)
     */
    public function actionAppointments()
    {
        $user = Yii::$app->user->identity;
        $masterInfo = MasterInfo::find()->where(['user_id' => $user->id])->one();

        $appointments = [];
        if ($masterInfo) {
            $appointments = Appointment::find()
                ->where(['master_info_id' => $masterInfo->id])
                ->orderBy(['start_time' => SORT_ASC])
                ->all();
        }

        return $this->render('appointments', [
            'appointments' => $appointments,
        ]);
    }

    public function actionChangeStatus()
    {
        Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;

        $appointmentId = Yii::$app->request->post('id');
        $status = Yii::$app->request->post('status');

        $appointment = Appointment::findOne($appointmentId);
        if (!$appointment) {
            return ['success' => false, 'message' => 'Запись не найдена'];
        }

        // Проверяем, что это запись текущего мастера
        $user = Yii::$app->user->identity;
        $masterInfo = MasterInfo::find()->where(['user_id' => $user->id])->one();

        if ($appointment->master_info_id != $masterInfo->id) {
            return ['success' => false, 'message' => 'Нет прав на изменение'];
        }

        $appointment->status = $status;
        if ($appointment->save()) {
            return ['success' => true, 'message' => 'Статус обновлен'];
        }

        return ['success' => false, 'message' => 'Ошибка сохранения'];
    }

    /**
     * Публичная страница мастера (доступна всем)
     */
    public function actionView($id)
    {
        // Находим мастера (пользователь с is_admin = 1)
        $user = User::find()
            ->where(['id' => $id, 'is_admin' => 1])
            ->one();

        if (!$user) {
            throw new \yii\web\NotFoundHttpException('Мастер не найден');
        }

        // Информация о мастере
        $masterInfo = MasterInfo::find()
            ->where(['user_id' => $user->id])
            ->one();

        // Если нет информации о мастере - создаем
        if (!$masterInfo) {
            $masterInfo = new MasterInfo();
            $masterInfo->user_id = $user->id;
            $masterInfo->is_active = 1;
            $masterInfo->save();
        }

        // Получаем услуги через master_services
        $services = Service::find()
            ->alias('s')
            ->select(['s.*', 'ms.price_modifier'])
            ->innerJoin('master_services ms', 'ms.service_id = s.id')
            ->where(['ms.master_info_id' => $masterInfo->id])
            ->andWhere(['s.is_active' => 1])
            ->all();

        // Расписание работы
        $schedule = MasterSchedule::find()
            ->where(['master_info_id' => $masterInfo->id])
            ->indexBy('day_of_week')
            ->all();

        // Портфолио
        $portfolio = MasterPortfolio::find()
            ->where(['master_info_id' => $masterInfo->id])
            ->orderBy(['created_at' => SORT_DESC])
            ->all();

        return $this->render('view', [
            'user' => $user,
            'masterInfo' => $masterInfo,
            'services' => $services,
            'schedule' => $schedule,
            'portfolio' => $portfolio,
        ]);
    }

    /**
     * Портфолио мастера (фото работ)
     */


    /**
     * Загрузка нового фото в портфолио
     */
    public function actionUploadPortfolio()
    {
        Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;

        try {
            $user = Yii::$app->user->identity;
            $masterInfo = MasterInfo::find()->where(['user_id' => $user->id])->one();

            if (!$masterInfo) {
                return ['success' => false, 'message' => 'Информация о мастере не найдена'];
            }

            // Загружаем файл
            $file = UploadedFile::getInstanceByName('imageFile');
            if (!$file) {
                return ['success' => false, 'message' => 'Файл не получен'];
            }

            $description = Yii::$app->request->post('description', '');

            // Проверяем расширение
            $ext = $file->extension;
            if (!in_array(strtolower($ext), ['jpg', 'jpeg', 'png', 'webp'])) {
                return ['success' => false, 'message' => 'Недопустимый формат файла'];
            }

            // Создаем папку
            $uploadPath = 'uploads/portfolio/' . $user->id;
            if (!is_dir($uploadPath)) {
                mkdir($uploadPath, 0777, true);
            }

            // Генерируем имя файла
            $fileName = 'photo_' . time() . '_' . rand(1000, 9999) . '.' . $ext;
            $filePath = $uploadPath . '/' . $fileName;

            if ($file->saveAs($filePath)) {
                $model = new MasterPortfolio();
                $model->master_info_id = $masterInfo->id;
                $model->photo_url = $filePath;
                $model->description = $description;

                if ($model->save()) {
                    return [
                        'success' => true,
                        'message' => 'Фото добавлено',
                        'photo' => [
                            'id' => $model->id,
                            'url' => '/' . $filePath,
                            'description' => $model->description,
                            'date' => date('d.m.Y'),
                        ]
                    ];
                } else {
                    return ['success' => false, 'message' => 'Ошибка сохранения в БД'];
                }
            } else {
                return ['success' => false, 'message' => 'Ошибка сохранения файла'];
            }
        } catch (\Exception $e) {
            return ['success' => false, 'message' => 'Ошибка: ' . $e->getMessage()];
        }
    }


    /**
     * Удаление фото из портфолио
     */
    public function actionDeletePortfolio()
    {
        Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;

        $id = Yii::$app->request->post('id');
        $user = Yii::$app->user->identity;

        $portfolio = MasterPortfolio::findOne($id);
        if (!$portfolio) {
            return ['success' => false, 'message' => 'Фото не найдено'];
        }

        // Проверяем, что это фото принадлежит текущему мастеру
        $masterInfo = MasterInfo::find()->where(['user_id' => $user->id])->one();
        if ($portfolio->master_info_id != $masterInfo->id) {
            return ['success' => false, 'message' => 'Нет прав на удаление'];
        }

        // Удаляем файл
        if (file_exists($portfolio->photo_url)) {
            unlink($portfolio->photo_url);
        }

        // Удаляем запись из БД
        if ($portfolio->delete()) {
            return ['success' => true, 'message' => 'Фото удалено'];
        }

        return ['success' => false, 'message' => 'Ошибка удаления'];
    }

    /**
     * Редактирование описания фото
     */
    public function actionUpdatePortfolioDescription()
    {
        Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;

        $id = Yii::$app->request->post('id');
        $description = Yii::$app->request->post('description');
        $user = Yii::$app->user->identity;

        $portfolio = MasterPortfolio::findOne($id);
        if (!$portfolio) {
            return ['success' => false, 'message' => 'Фото не найдено'];
        }

        // Проверяем права
        $masterInfo = MasterInfo::find()->where(['user_id' => $user->id])->one();
        if ($portfolio->master_info_id != $masterInfo->id) {
            return ['success' => false, 'message' => 'Нет прав на редактирование'];
        }

        $portfolio->description = $description;
        if ($portfolio->save()) {
            return ['success' => true, 'message' => 'Описание обновлено'];
        }

        return ['success' => false, 'message' => 'Ошибка сохранения'];
    }
}